/**
\file
\brief 	This is the Glucose Meter sowftware class. It executes a blood glucose measurement
		using the amperometric method.
\author Samuel Quiroz
\date 2010-09-11
*/

#ifndef _GLUCOSE_METER
#define _GLUCOSE_METER


#include <hidef.h> /* for EnableInterrupts macro */
#include "derivative.h" /* include peripheral declarations */
#include "PublicTypes.h"
#include "ADC.h"
#include "AverageFilter.h"
#include "SwTimer.h"


/* Glucose meter parameters */

////////////////////////////
//#define GLU_DEBUG		//define this to send real time data to PC for debug
//////////////////////


/* All timers are in ms */

/** The time in ms that will elapse between the blood is detected and the current is measured */
#define FINISH_GLUCOSE_MEASUREMENT_TIME		5000

/** The sampling period where the strip current will be measured */
#define GLUCOSE_SAMPLING_PERIOD				50	//ms


/* Pin definitions */

/** The ADC channel of Working Electrode 1 */
#define ADC_CHANNEL_WORKING_ELECTRODE1	8

/** The ADC channel of Working Electrode 2 */
#define ADC_CHANNEL_WORKING_ELECTRODE2	0

/** The voltage blood detection voltage threshold in mV */
#define BLOOD_DETECT_THRESHOLD_mV		560		//mV corresponds to 1uA approx

/** ADC Reference voltage in mV */
#define REF_VOLTAGE_mV					3300	//mV

/** ADC resolution in bits */
#define GLU_ADC_RESOLUTION					255

/** The voltage blood detection voltage threshold calculated in ADC counts */
#define BLOOD_DETECT_THRESHOLD		(UINT32)(((UINT32)BLOOD_DETECT_THRESHOLD_mV*(UINT32)GLU_ADC_RESOLUTION)/(UINT32)REF_VOLTAGE_mV)     ///*43.27

/** Voltage Offset of the electrodes */
#define	ELECTRODES_VOLTAGE_OFFSET_mV			100		//mV

/** Same but in ADC counts */
//#define	ELECTRODES_VOLTAGE_OFFSET			(UINT32)(((UINT32)ELECTRODES_VOLTAGE_OFFSET_mV*(UINT32)GLU_ADC_RESOLUTION)/(UINT32)REF_VOLTAGE_mV)
#define	ELECTRODES_VOLTAGE_OFFSET			37


/** In debug mode, array size used to send data to PC */
#define GLU_GRAPH_DATA_ARRAY_SIZE	16


/** Starts a glucose measurement 
	\returns TRUE if the measurement was started, FALSE if not
*/
UINT8 GlucoseMeter_StartMeasurement(void);

/** Abort a glucose measurement */
void GlucoseMeter_AbortMeasurement(void);

/** The Periodic task that must be called from main */
void GlucoseMeter_PeriodicTask(void);

/** Call this function once on initialization */
void GlucoseMeter_Init(void);

/** 
	Starts a glucose calibration routine 
	\param knownGlucoseValue A known glucose vale in mgr/dl
	\returns TRUE if the measurement was started, FALSE if not
*/
UINT8 GlucoseMeter_StartCalibration(UINT16 knownGlucoseValue);

/** The result of the glucose measurment in mgr/dl */
extern UINT16 GlucoseMeter_GlucoseValue;

/** The calibration constant for the */
extern INT16 GlucoseMeter_CalibrationConstant;

/** Array of functions for the Glucose Events. Must be defined in upper layer */
extern const pFunc_t GlucoseMeter_Events[];	


/** Array used in debug mode to send graph data to PC */
extern UINT8 GlucoseMeter_GraphDataToPc[GLU_GRAPH_DATA_ARRAY_SIZE];


typedef enum
{
	EVENT_GLU_NONE,						/**< No event */
	EVENT_GLU_BLOOD_DETECTED,			/**< Blood was detected */
	EVENT_GLU_FINISH_MEASUREMENT_OK,	/**< Measurement finished correctly */
	EVENT_GLU_FINISH_CALIBRATION_OK,	/**< Calibration finished correctly */
	EVENT_GLU_DEBUG_NEW_DATA_READY,		/**< Send real time ADC data for dedub */
} GlucoseMeter_Event_e;

#endif //_GLUCOSE_METER